% Copyright (c) 2016 Giampaolo D'Alessandro
% Mathematical Sciences - University of Southampton
% Email: dales@soton.ac.uk
% 
% Permission is hereby granted, free of charge, to any person obtaining a copy
% of this software and associated documentation files (the "Software"), to deal
% in the Software without restriction, including without limitation the rights
% to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
% copies of the Software, and to permit persons to whom the Software is
% furnished to do so, subject to the following conditions:
% 
% The above copyright notice and this permission notice shall be included in
% all copies or substantial portions of the Software.
% 
% THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
% IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
% FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
% AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
% LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
% OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
% SOFTWARE.

function [Diff_eff] = RamanNath(lambda,x,y,z,T,theta,alpha_tot,q,sigma_p,allow_fig)
% Function to compute the diffraction pattern generated by a temperature
% profile in a liquid crystal cell in the Raman-Nath regime.
%
% Input parameters
% lambda (scalar) - The probe wavelength
% x,y,z (vectors of size nx, ny and nz) - The coordinate points [in meters]
% T (array of size (ny,nx,nz)) - The temperature profile [in Kelvin]
% theta (array of size (nz,1) - The liquid crystal alignment in the cell.
% q (scalar) - The grating wave number
% sigma_p - half-width of the probe beam (in meters)
% allow_fig - Flag to print the figures

% Specify the probe wavenumber
k = 2*pi/lambda;
% Specify the number of grid points in each direction
[ny,nx,nz] = size(T);
% Specify the cell thickness and width
L = max(z)-min(z);
LX = max(x)-min(x)+x(2)-x(1);
LY = max(y)-min(y)+y(2)-y(1);
% Ambient temperature (in Kelvin)
T_A = 273.16 + 22;
Probe_beam_angleInDegrees = 15;
Probe_beam_angle = degtorad(Probe_beam_angleInDegrees);

% Plot parameters
FontName='Times';
FontSize=24;
FontSize_SP = 12;
FontAngle='normal'; %'italic';
LineWidth=2;
% MarkerSize=12;  % In points (72 points per inch)

% Refractive index data.
% Data from Li et al, IEEE J. Disp. Tech 1(1), 51 (2005)
% The refractive indices of interest are the extraordinary and ordinary
% refractive indices of E7.

% We specify the refractive indices measured at a set of temperatures and
% do a spline interpolation between them.
ne_m = [...
  [1.8170 1.8084 1.8005 1.7909 1.7811 1.7695 1.7549 1.7355 1.6936]; ...
  [1.7921 1.7847 1.7763 1.7679 1.7581 1.7472 1.7333 1.7154 1.6779]; ...
  [1.7664 1.7589 1.7512 1.7433 1.7344 1.7237 1.7109 1.6941 1.6601]; ...
  [1.7542 1.7466 1.7394 1.7317 1.7231 1.7124 1.7001 1.6837 1.6511]; ...
  [1.7446 1.7378 1.7305 1.7229 1.7142 1.7037 1.6919 1.6761 1.6440]; ...  
  [1.7407 1.7338 1.7263 1.7190 1.7102 1.7001 1.6882 1.6721 1.6405]; ...
  ];
no_m = [...
  [1.5435 1.5431 1.5424 1.5430 1.5448 1.5470 1.5491 1.5538 1.5690]; ...
  [1.5360 1.5357 1.5353 1.5357 1.5369 1.5383 1.5406 1.5449 1.5588]; ...
  [1.5273 1.5269 1.5268 1.5271 1.5277 1.5287 1.5308 1.5350 1.5479]; ...
  [1.5230 1.5227 1.5225 1.5226 1.5231 1.5239 1.5261 1.5299 1.5428]; ...
  [1.5197 1.5188 1.5189 1.5189 1.5191 1.5205 1.52221 1.526 1.5377]; ...
  [1.5179 1.5177 1.5176 1.5177 1.5179 1.5189 1.5206 1.5246 1.5353]; ...
  ];

lambda_m = [450; 486; 546; 589; 633; 656]*1e-9; % in m
T_m = 273.15 + (15:5:55);  % in K
[LAMBDA_M,T_M] = ndgrid(lambda_m,T_m);
n_e = griddedInterpolant(LAMBDA_M,T_M,ne_m,'spline');
n_o = griddedInterpolant(LAMBDA_M,T_M,no_m,'spline');
 
% Absorbance data - Here too we perform a spline interpolation.
% alpha_m = [166 139 110 90 74 70]*1e2;  % in m^{-1}
% alpha = griddedInterpolant(lambda_m,alpha_m,'spline');

%% Compute and plot the z-average of the refractive index
% We use a clencurt method to compute the total phase shift
% g = n_e(lambda*ones(size(T)),T+T_A);
g = n_eff(theta,T+T_A,lambda,n_e,n_o);
[~,w] = clencurt(nz-1);
n_av = 0.5*multiprod(g,w',3,1);

[X,Y] = meshgrid(x,y);
if (allow_fig)
  figure(3);clf;
  set(gcf,'PaperOrientation','Landscape','PaperType','A4',...
    'PaperUnits','normalized','PaperPosition',[0 0 1 1]);
  axes('Parent',gcf,...
    'YMinorTick','on','XMinorTick','on',...
    'FontSize',FontSize,'FontName',FontName,'FontAngle',FontAngle);
  view(gca,[27 36]);
  grid('on');
  hold(gca,'all');
  surf(X,Y,n_av);
  xlabel('x');
  ylabel('y');
  title('Average refractive index');
  set(gcf,'Renderer','Zbuffer');
  axis('tight');

  figure(4);clf;
  set(gcf,'PaperOrientation','Landscape','PaperType','A4',...
    'PaperUnits','normalized','PaperPosition',[0 0 1 1]);
  axes('Parent',gcf,...
    'YMinorTick','on','XMinorTick','on',...
    'FontSize',FontSize,'FontName',FontName,'FontAngle',FontAngle);
  plot(x,n_av(floor(0.5*ny)+1,:),'r-','LineWidth',LineWidth);
  xlabel('x');
  title('n_{av} at y=0');
  axis('tight');
end

%% Compute the beam at output of the cell

Beam = exp(-(X.^2+Y.^2)/sigma_p^2).*exp(1i*k*n_av*L/cos(Probe_beam_angle));

if (allow_fig)
  figure(4);clf;
  set(gcf,'PaperOrientation','Landscape','PaperType','A4',...
    'PaperUnits','normalized','PaperPosition',[0 0 1 1]);
  axes('Parent',gcf,...
    'YMinorTick','on','XMinorTick','on',...
    'FontSize',FontSize,'FontName',FontName,'FontAngle',FontAngle);
  view(gca,[27 36]);
  grid('on');
  hold(gca,'all');
  surf(X,Y,real(Beam));
  xlabel('x');
  ylabel('y');
  title('Beam profile (real part)');
  set(gcf,'Renderer','Zbuffer');
  axis('tight');

  figure(5);clf;
  set(gcf,'PaperOrientation','Landscape','PaperType','A4',...
    'PaperUnits','normalized','PaperPosition',[0 0 1 1]);
  axes('Parent',gcf,...
    'YMinorTick','on','XMinorTick','on',...
    'FontSize',FontSize,'FontName',FontName,'FontAngle',FontAngle);
  plot(x,real(Beam(floor(0.5*ny)+1,:)),'r-','LineWidth',LineWidth);
  xlabel('x');
  title('Beam profile (real part) at y=0');
  axis('tight');
end

%% Compute the diffracted beams

Diffract = fftshift(fft2(Beam));
kx = (-round(0.5*nx):(round(0.5*nx)-1))*2*pi/LX;
ky = (-round(0.5*ny):(round(0.5*ny)-1))*2*pi/LY;
[KX,KY] = meshgrid(kx,ky);

if (allow_fig)
  figure(6);clf;
  set(gcf,'PaperOrientation','Landscape','PaperType','A4',...
    'PaperUnits','normalized','PaperPosition',[0 0 1 1]);
  axes('Parent',gcf,...
    'YMinorTick','on','XMinorTick','on',...
    'FontSize',FontSize,'FontName',FontName,'FontAngle',FontAngle);
  view(gca,[27 36]);
  grid('on');
  hold(gca,'all');
  surf(KX,KY,log(abs(Diffract).^2));
  xlabel('k_x');
  ylabel('k_y');
  title('Diffracted beam intensity');
  set(gcf,'Renderer','Zbuffer');
  axis('tight');
end

%% Compute and plot the diffraction patterns and efficiency
% We compute the total power in the zero and first diffracted order by
% summing the energy in a circle of radius 0.4*q around each order.

% Fundamental
j0 = find(KX.^2+KY.^2<(0.4*q)^2);
[j0x,j0y] = ind2sub(size(KX),j0);
P0 = sum(abs(Diffract(j0)).^2);
% +1 diffracted order
jp1 = find((KX-q).^2+KY.^2<(0.4*q)^2);
[jp1x,jp1y] = ind2sub(size(KX),jp1);
Pp1 = sum(abs(Diffract(jp1)).^2);
% -1 diffracted order
jm1 = find((KX+q).^2+KY.^2<(0.4*q)^2);
% [jm1x,jm1y] = ind2sub(size(KX),jm1);
Pm1 = sum(abs(Diffract(jm1)).^2); %#ok<FNDSB>
% Compute the relative power between +/-1 and fundamental
% Relative_Power = 0.5*(Pp1+Pm1)/P0;
% Compute the diffraction efficiency.  In doing so, correct the total power
% by dividing by the absorption coefficient during propagation through the
% cell. 
Diff_eff = 0.5*(Pp1+Pm1)/((P0+Pp1+Pm1)*exp(alpha_tot));

if (allow_fig)
  figure(7);clf;
  set(gcf,'PaperOrientation','Landscape','PaperType','A4',...
    'PaperUnits','normalized','PaperPosition',[0 0 1 1]);

  subplot(2,2,1,'Parent',gcf,...
    'YMinorTick','on','XMinorTick','on',...
    'FontSize',FontSize_SP,'FontName',FontName,'FontAngle',FontAngle);
  view(gca,[27 36]);
  grid('on');
  hold(gca,'all');
  surf(KX,KY,abs(Diffract));
  xlabel('k_x');
  ylabel('k_y');
  set(gcf,'Renderer','Zbuffer');
  axis('tight');
  
  subplot(2,2,2,'Parent',gcf,...
    'YMinorTick','on','XMinorTick','on',...
    'FontSize',FontSize_SP,'FontName',FontName,'FontAngle',FontAngle);
  % grid('off');
  % hold(gca,'all');
  imagesc([kx(1),kx(end)],[ky(1),ky(end)],log10(max(abs(Diffract),1e-4)));
  colorbar;

  subplot(2,2,3,'Parent',gcf,...
    'YMinorTick','on','XMinorTick','on',...
    'FontSize',FontSize_SP,'FontName',FontName,'FontAngle',FontAngle);
  view(gca,[27 36]);
  grid('on');
  hold(gca,'all');
  surf(KX(j0x,j0y),KY(j0x,j0y),abs(Diffract(j0x,j0y)));
  xlabel('k_x');
  ylabel('k_y');
  set(gcf,'Renderer','Zbuffer');
  axis('tight');

  subplot(2,2,4,'Parent',gcf,...
    'YMinorTick','on','XMinorTick','on',...
    'FontSize',FontSize_SP,'FontName',FontName,'FontAngle',FontAngle);
  view(gca,[27 36]);
  grid('on');
  hold(gca,'all');
  surf(KX(jp1x,jp1y),KY(jp1x,jp1y),abs(Diffract(jp1x,jp1y)));
  xlabel('k_x');
  ylabel('k_y');
  set(gcf,'Renderer','Zbuffer');
  axis('tight');
end

%% Write the final output
display(sprintf('The diffraction efficiency at %.1f nm is %.4f %%', ...
  lambda*1e9,Diff_eff*100));
end

function N_EFF = n_eff(theta,T,lambda,n_e,n_o)
% Returns the effective refractive index in a cell with z-dependent
% alignment given by the (nzx1) array theta and the temperature profile T,
% an array of size (nx,ny,nz).
%
% The formula for the effective refractive index is
% %
% \begin{equation*}
%   n_eff(\theta,T) = \frac{n_{o}(T) n_{e}(T)}
%     {\sqrt{n_{e}^{2}(T) \sin^{2}(\theta) + 
%            n_{o}^{2}(T) \cos^{2}(\theta)}} \, .
% \end{equation*}

N_E = n_e(lambda*ones(size(T)),T);
N_O = n_o(lambda*ones(size(T)),T);
N_EFF = N_E.*N_O;
for k = 1:length(theta)
  N_EFF(:,:,k) = N_EFF(:,:,k)./sqrt(sin(theta(k))^2*N_E(:,:,k).^2 + ...
                                    cos(theta(k))^2*N_O(:,:,k).^2);
end
end